<?php
/**
 * @subpackage  mod_xiroweb_articles_focus
 *
 * @copyright   (C) 2021-2022 Xiroweb All rights reserved.
 * @copyright   (C) 2010 Open Source Matters, Inc. <https://www.joomla.org>
 * 
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Xiroweb\Module\XirowebArticlesFocus\Site\Helper;

\defined('_JEXEC') or die;

use Joomla\CMS\Access\Access;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Router\Route;
use Joomla\Component\Content\Administrator\Extension\ContentComponent;
use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\Utilities\ArrayHelper;

/**
 * Helper for mod_xiroweb_articles_focus
 *
 * @since  1.6
 */
abstract class XirowebArticlesFocusHelper
{
	/**
	 * Get a list of articles from a specific category
	 *
	 * @param   \Joomla\Registry\Registry  &$params  object holding the models parameters
	 *
	 * @return  mixed
	 *
	 * @since  1.6
	 */
	public static function getList(&$params)
	{
		$app     = Factory::getApplication();
		$factory = $app->bootComponent('com_content')->getMVCFactory();

		// Get an instance of the generic articles model
		$articles = $factory->createModel('Articles', 'Site', ['ignore_request' => true]);

		// Set application parameters in model
		$input     = $app->input;
		$appParams = $app->getParams();
		$articles->setState('params', $appParams);
		$articles->setState('list.start', 0);
		$articles->setState('filter.published', ContentComponent::CONDITION_PUBLISHED);
		// Access filter
		$access     = !ComponentHelper::getParams('com_content')->get('show_noauth');
		$authorised = Access::getAuthorisedViewLevels(Factory::getUser()->get('id'));
		$articles->setState('filter.access', $access);
        
        if ($params->get('switchmode', '') == 'ids') {
                
            $included_articles = (array) $params->get('included_articles') ?: array();
            $included_articles_ids =  ArrayHelper::getColumn($included_articles, 'article_id');
            $included_articles_ids = array_filter($included_articles_ids);
            $included_articles_ids = array_unique($included_articles_ids);

            if ($included_articles_ids)
            {
                $articles->setState('filter.article_id', $included_articles_ids);
                $articles->setState('filter.article_id.include', true);
            } else {
                return;
            }

            $articleIds = ArrayHelper::toInteger($included_articles_ids);
            
            $articles->setState('list.ordering', '');
            $articles->setState('list.direction', 'FIELD(a.id, '. implode(',',$articleIds) .')');
    
        } elseif ($params->get('switchmode', '') == 'options') {
    
            $articles->setState('list.limit', (int) $params->get('count', 0));
            $catids = $params->get('catid');
            $articles->setState('filter.category_id.include', (bool) $params->get('category_filtering_type', 1));
            $articles->setState('filter.category_id', $catids);

            $ordering = $params->get('article_ordering', 'a.ordering');
            $ordering = $params->get('article_ordering', 'a.ordering');

            switch ($ordering)
            {
                case 'random':
                    $articles->setState('list.ordering', Factory::getDbo()->getQuery(true)->rand());
                    break;

                case 'rating_count':
                case 'rating':
                    $articles->setState('list.ordering', $ordering);
                    $articles->setState('list.direction', $params->get('article_ordering_direction', 'ASC'));

                    if (!PluginHelper::isEnabled('content', 'vote'))
                    {
                        $articles->setState('list.ordering', 'a.ordering');
                    }

                    break;

                default:
                    $articles->setState('list.ordering', $ordering);
                    $articles->setState('list.direction', $params->get('article_ordering_direction', 'ASC'));
                    break;
            }
            $articles->setState('filter.featured', $params->get('show_front', 'show'));

            
        } else {
            return;
        }
        // Filter by language
        $articles->setState('filter.language', $app->getLanguageFilter());


		$items = $articles->getItems();

		// Display options
		$show_date        = $params->get('show_date', 0);
		$show_date_field  = $params->get('show_date_field', 'created');
		$show_date_format = $params->get('show_date_format', 'Y-m-d H:i:s');
		$show_category    = $params->get('show_category', 0);
		$show_hits        = $params->get('show_hits', 0);
		$show_author      = $params->get('show_author', 0);
		$show_introtext   = $params->get('show_introtext', 0);
		$introtext_limit  = $params->get('introtext_limit', 160);

		// Find current Article ID if on an article page
		$option = $input->get('option');
		$view   = $input->get('view');

		if ($option === 'com_content' && $view === 'article')
		{
			$active_article_id = $input->getInt('id');
		}
		else
		{
			$active_article_id = 0;
		}

		// Prepare data for display using display options
		foreach ($items as &$item)
		{
			$item->slug = $item->id . ':' . $item->alias;

			if ($access || \in_array($item->access, $authorised))
			{
				// We know that user has the privilege to view the article
				$item->link = Route::_(RouteHelper::getArticleRoute($item->slug, $item->catid, $item->language));
			}
			else
			{
				$menu      = $app->getMenu();
				$menuitems = $menu->getItems('link', 'index.php?option=com_users&view=login');

				if (isset($menuitems[0]))
				{
					$Itemid = $menuitems[0]->id;
				}
				elseif ($input->getInt('Itemid') > 0)
				{
					// Use Itemid from requesting page only if there is no existing menu
					$Itemid = $input->getInt('Itemid');
				}

				$item->link = Route::_('index.php?option=com_users&view=login&Itemid=' . $Itemid);
			}

			// Used for styling the active article
			$item->active      = $item->id == $active_article_id ? 'active' : '';
			$item->displayDate = '';

			if ($show_date)
			{
				$item->displayDate = HTMLHelper::_('date', $item->$show_date_field, $show_date_format);
			}

			if ($item->catid)
			{
				$item->displayCategoryLink  = Route::_(RouteHelper::getCategoryRoute($item->catid, $item->category_language));
				$item->displayCategoryTitle = $show_category ? '<a href="' . $item->displayCategoryLink . '">' . $item->category_title . '</a>' : '';
			}
			else
			{
				$item->displayCategoryTitle = $show_category ? $item->category_title : '';
			}

			$item->displayHits       = $show_hits ? $item->hits : '';
			$item->displayAuthorName = $show_author ? $item->author : '';

			if ($show_introtext)
			{
				$item->introtext = HTMLHelper::_('content.prepare', $item->introtext, '', 'mod_xiroweb_articles_focus.content');
				$item->introtext = self::_cleanIntrotext($item->introtext);
			}

			$item->displayIntrotext = $show_introtext ? self::truncate($item->introtext, $introtext_limit) : '';
			$item->displayReadmore  = $item->alternative_readmore;
		}

		return $items;
	}



	/**
	 * Strips unnecessary tags from the introtext
	 *
	 * @param   string  $introtext  introtext to sanitize
	 *
	 * @return mixed|string
	 *
	 * @since  1.6
	 */
	public static function _cleanIntrotext($introtext)
	{
		$introtext = str_replace(array('<p>', '</p>'), ' ', $introtext);
		$introtext = strip_tags($introtext, '<a><em><strong><joomla-hidden-mail>');
		$introtext = trim($introtext);

		return $introtext;
	}

	/**
	 * Method to truncate introtext
	 *
	 * The goal is to get the proper length plain text string with as much of
	 * the html intact as possible with all tags properly closed.
	 *
	 * @param   string   $html       The content of the introtext to be truncated
	 * @param   integer  $maxLength  The maximum number of characters to render
	 *
	 * @return  string  The truncated string
	 *
	 * @since   1.6
	 */
	public static function truncate($html, $maxLength = 0)
	{
		$baseLength = \strlen($html);

		// First get the plain text string. This is the rendered text we want to end up with.
		$ptString = HTMLHelper::_('string.truncate', $html, $maxLength, $noSplit = true, $allowHtml = false);

		for ($maxLength; $maxLength < $baseLength;)
		{
			// Now get the string if we allow html.
			$htmlString = HTMLHelper::_('string.truncate', $html, $maxLength, $noSplit = true, $allowHtml = true);

			// Now get the plain text from the html string.
			$htmlStringToPtString = HTMLHelper::_('string.truncate', $htmlString, $maxLength, $noSplit = true, $allowHtml = false);

			// If the new plain text string matches the original plain text string we are done.
			if ($ptString === $htmlStringToPtString)
			{
				return $htmlString;
			}

			// Get the number of html tag characters in the first $maxlength characters
			$diffLength = \strlen($ptString) - \strlen($htmlStringToPtString);

			// Set new $maxlength that adjusts for the html tags
			$maxLength += $diffLength;

			if ($baseLength <= $maxLength || $diffLength <= 0)
			{
				return $htmlString;
			}
		}

		return $html;
	}

	
}
